/*
 * Decompiled with CFR 0.152.
 */
package javaxt.sql;

import java.io.PrintWriter;
import java.sql.SQLException;
import java.util.LinkedList;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import javax.sql.ConnectionEvent;
import javax.sql.ConnectionEventListener;
import javax.sql.ConnectionPoolDataSource;
import javax.sql.PooledConnection;
import javaxt.sql.Connection;
import javaxt.sql.Database;

public class ConnectionPool {
    private ConnectionPoolDataSource dataSource;
    private int maxConnections;
    private long timeoutMs;
    private PrintWriter logWriter;
    private Semaphore semaphore;
    private PoolConnectionEventListener poolConnectionEventListener;
    private LinkedList<PooledConnection> recycledConnections;
    private int activeConnections;
    private boolean isDisposed;
    private boolean doPurgeConnection;
    private PooledConnection connectionInTransition;

    public ConnectionPool(Database database, int maxConnections) throws SQLException {
        this(database.getConnectionPoolDataSource(), maxConnections, 60);
    }

    public ConnectionPool(Database database, int maxConnections, int timeout) throws SQLException {
        this(database.getConnectionPoolDataSource(), maxConnections, timeout);
    }

    public ConnectionPool(ConnectionPoolDataSource dataSource, int maxConnections) {
        this(dataSource, maxConnections, 60);
    }

    public ConnectionPool(ConnectionPoolDataSource dataSource, int maxConnections, int timeout) {
        this.dataSource = dataSource;
        this.maxConnections = maxConnections;
        this.timeoutMs = (long)timeout * 1000L;
        try {
            this.logWriter = dataSource.getLogWriter();
        }
        catch (SQLException e) {
            // empty catch block
        }
        if (maxConnections < 1) {
            throw new IllegalArgumentException("Invalid maxConnections value.");
        }
        this.semaphore = new Semaphore(maxConnections, true);
        this.recycledConnections = new LinkedList();
        this.poolConnectionEventListener = new PoolConnectionEventListener();
    }

    public synchronized void close() throws SQLException {
        if (this.isDisposed) {
            return;
        }
        this.isDisposed = true;
        SQLException e = null;
        while (!this.recycledConnections.isEmpty()) {
            PooledConnection pconn = this.recycledConnections.remove();
            try {
                pconn.close();
            }
            catch (SQLException e2) {
                if (e != null) continue;
                e = e2;
            }
        }
        if (e != null) {
            throw e;
        }
    }

    public Connection getConnection() throws SQLException {
        return new Connection(this.getConnection2(this.timeoutMs));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private java.sql.Connection getConnection2(long timeoutMs) throws SQLException {
        ConnectionPool connectionPool = this;
        synchronized (connectionPool) {
            if (this.isDisposed) {
                throw new IllegalStateException("Connection pool has been disposed.");
            }
        }
        try {
            if (!this.semaphore.tryAcquire(timeoutMs, TimeUnit.MILLISECONDS)) {
                throw new TimeoutException();
            }
        }
        catch (InterruptedException e) {
            throw new RuntimeException("Interrupted while waiting for a database connection.", e);
        }
        boolean ok = false;
        try {
            java.sql.Connection conn = this.getConnection3();
            ok = true;
            java.sql.Connection connection = conn;
            return connection;
        }
        finally {
            if (!ok) {
                this.semaphore.release();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private synchronized java.sql.Connection getConnection3() throws SQLException {
        java.sql.Connection conn;
        PooledConnection pconn;
        if (this.isDisposed) {
            throw new IllegalStateException("Connection pool has been disposed.");
        }
        if (!this.recycledConnections.isEmpty()) {
            pconn = this.recycledConnections.remove();
        } else {
            pconn = this.dataSource.getPooledConnection();
            pconn.addConnectionEventListener(this.poolConnectionEventListener);
        }
        try {
            this.connectionInTransition = pconn;
            conn = pconn.getConnection();
        }
        finally {
            this.connectionInTransition = null;
        }
        ++this.activeConnections;
        this.assertInnerState();
        return conn;
    }

    public Connection getValidConnection() {
        long time = System.currentTimeMillis();
        long timeoutTime = time + this.timeoutMs;
        int triesWithoutDelay = this.getInactiveConnections() + 1;
        do {
            java.sql.Connection conn;
            if ((conn = this.getValidConnection2(time, timeoutTime)) != null) {
                return new Connection(conn);
            }
            if (--triesWithoutDelay > 0) continue;
            triesWithoutDelay = 0;
            try {
                Thread.sleep(250L);
            }
            catch (InterruptedException e) {
                throw new RuntimeException("Interrupted while waiting for a valid database connection.", e);
            }
        } while ((time = System.currentTimeMillis()) < timeoutTime);
        throw new TimeoutException("Timeout while waiting for a valid database connection.");
    }

    private java.sql.Connection getValidConnection2(long time, long timeoutTime) {
        java.sql.Connection conn;
        long rtime = Math.max(1L, timeoutTime - time);
        try {
            conn = this.getConnection2(rtime);
        }
        catch (SQLException e) {
            return null;
        }
        rtime = timeoutTime - System.currentTimeMillis();
        int rtimeSecs = Math.max(1, (int)((rtime + 999L) / 1000L));
        try {
            if (conn.isValid(rtimeSecs)) {
                return conn;
            }
        }
        catch (SQLException e) {
            // empty catch block
        }
        this.purgeConnection(conn);
        return null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private synchronized void purgeConnection(java.sql.Connection conn) {
        try {
            this.doPurgeConnection = true;
            conn.close();
        }
        catch (SQLException sQLException) {
        }
        finally {
            this.doPurgeConnection = false;
        }
    }

    private synchronized void recycleConnection(PooledConnection pconn) {
        if (this.isDisposed || this.doPurgeConnection) {
            this.disposeConnection(pconn);
            return;
        }
        if (this.activeConnections <= 0) {
            throw new AssertionError();
        }
        --this.activeConnections;
        this.semaphore.release();
        this.recycledConnections.add(pconn);
        this.assertInnerState();
    }

    private synchronized void disposeConnection(PooledConnection pconn) {
        pconn.removeConnectionEventListener(this.poolConnectionEventListener);
        if (!this.recycledConnections.remove(pconn) && pconn != this.connectionInTransition) {
            if (this.activeConnections <= 0) {
                throw new AssertionError();
            }
            --this.activeConnections;
            this.semaphore.release();
        }
        this.closeConnectionAndIgnoreException(pconn);
        this.assertInnerState();
    }

    private void closeConnectionAndIgnoreException(PooledConnection pconn) {
        try {
            pconn.close();
        }
        catch (SQLException e) {
            this.log("Error while closing database connection: " + e.toString());
        }
    }

    private void log(String msg) {
        String s = "ConnectionPool: " + msg;
        try {
            if (this.logWriter == null) {
                System.err.println(s);
            } else {
                this.logWriter.println(s);
            }
        }
        catch (Exception exception) {
            // empty catch block
        }
    }

    private synchronized void assertInnerState() {
        if (this.activeConnections < 0) {
            throw new AssertionError();
        }
        if (this.activeConnections + this.recycledConnections.size() > this.maxConnections) {
            throw new AssertionError();
        }
        if (this.activeConnections + this.semaphore.availablePermits() > this.maxConnections) {
            throw new AssertionError();
        }
    }

    public synchronized int getActiveConnections() {
        return this.activeConnections;
    }

    public synchronized int getInactiveConnections() {
        return this.recycledConnections.size();
    }

    public int getMaxConnections() {
        return this.maxConnections;
    }

    public ConnectionPoolDataSource getConnectionPoolDataSource() {
        return this.dataSource;
    }

    public int getTimeout() {
        return Math.round(this.timeoutMs / 1000L);
    }

    private class PoolConnectionEventListener
    implements ConnectionEventListener {
        private PoolConnectionEventListener() {
        }

        @Override
        public void connectionClosed(ConnectionEvent event) {
            PooledConnection pconn = (PooledConnection)event.getSource();
            ConnectionPool.this.recycleConnection(pconn);
        }

        @Override
        public void connectionErrorOccurred(ConnectionEvent event) {
            PooledConnection pconn = (PooledConnection)event.getSource();
            ConnectionPool.this.disposeConnection(pconn);
        }
    }

    public static class TimeoutException
    extends RuntimeException {
        private static final long serialVersionUID = 1L;

        public TimeoutException() {
            super("Timeout while waiting for a free database connection.");
        }

        public TimeoutException(String msg) {
            super(msg);
        }
    }
}

